<?php
require_once '../config.php';
requireLogin();

// Handle form submissions
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    // Handle bKash API configuration
    if (isset($_POST['save_bkash_config'])) {
        $apiUsername = sanitize($_POST['bkash_username'] ?? '');
        $apiPassword = sanitize($_POST['bkash_password'] ?? '');
        $apiAppKey = sanitize($_POST['bkash_app_key'] ?? '');
        $apiAppSecret = sanitize($_POST['bkash_app_secret'] ?? '');
        $isSandbox = isset($_POST['bkash_sandbox']) ? 1 : 0;
        $status = isset($_POST['bkash_status']) ? 1 : 0;
        
        try {
            // Check if bKash automatic gateway exists
            $stmt = $pdo->prepare("SELECT id FROM payment_gateways WHERE gateway_type = 'bkash' AND type = 'automatic'");
            $stmt->execute();
            $existing = $stmt->fetch();
            
            if ($existing) {
                // Update existing
                $stmt = $pdo->prepare("
                    UPDATE payment_gateways 
                    SET api_username = ?, api_password = ?, api_app_key = ?, api_app_secret = ?, is_sandbox = ?, status = ?
                    WHERE id = ?
                ");
                $stmt->execute([$apiUsername, $apiPassword, $apiAppKey, $apiAppSecret, $isSandbox, $status, $existing['id']]);
            } else {
                // Create new
                $stmt = $pdo->prepare("
                    INSERT INTO payment_gateways (name, type, gateway_type, api_username, api_password, api_app_key, api_app_secret, is_sandbox, status)
                    VALUES ('bKash Automatic', 'automatic', 'bkash', ?, ?, ?, ?, ?, ?)
                ");
                $stmt->execute([$apiUsername, $apiPassword, $apiAppKey, $apiAppSecret, $isSandbox, $status]);
            }
            
            setFlash('success', 'bKash API configuration saved successfully');
            header('Location: payment-gateways.php');
            exit;
        } catch(PDOException $e) {
            setFlash('error', 'Failed to save bKash configuration: ' . $e->getMessage());
        }
    }
    
    if (isset($_POST['add_gateway'])) {
        $name = sanitize($_POST['name'] ?? '');
        $type = $_POST['type'] ?? 'manual';
        $gatewayType = sanitize($_POST['gateway_type'] ?? '');
        $accountNumber = sanitize($_POST['account_number'] ?? '');
        $accountName = sanitize($_POST['account_name'] ?? '');
        $accountType = sanitize($_POST['account_type'] ?? '');
        $instructions = sanitize($_POST['instructions'] ?? '');
        
        try {
            $stmt = $pdo->prepare("
                INSERT INTO payment_gateways (name, type, gateway_type, account_number, account_name, account_type, instructions, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, 1)
            ");
            $stmt->execute([$name, $type, $gatewayType, $accountNumber, $accountName, $accountType, $instructions]);
            setFlash('success', 'Payment gateway added successfully');
            header('Location: payment-gateways.php');
            exit;
        } catch(PDOException $e) {
            setFlash('error', 'Failed to add gateway');
        }
    }
    
    if (isset($_POST['toggle_status'])) {
        $id = intval($_POST['gateway_id']);
        $status = intval($_POST['status']);
        
        try {
            $stmt = $pdo->prepare("UPDATE payment_gateways SET status = ? WHERE id = ?");
            $stmt->execute([$status, $id]);
            setFlash('success', 'Gateway status updated');
            header('Location: payment-gateways.php');
            exit;
        } catch(PDOException $e) {
            setFlash('error', 'Failed to update status');
        }
    }
    
    if (isset($_POST['delete_gateway'])) {
        $id = intval($_POST['gateway_id']);
        
        try {
            $stmt = $pdo->prepare("DELETE FROM payment_gateways WHERE id = ? AND type = 'manual'");
            $stmt->execute([$id]);
            setFlash('success', 'Gateway deleted successfully');
            header('Location: payment-gateways.php');
            exit;
        } catch(PDOException $e) {
            setFlash('error', 'Failed to delete gateway');
        }
    }
}

// Get all gateways
try {
    $stmt = $pdo->query("SELECT * FROM payment_gateways ORDER BY sort_order ASC, id DESC");
    $gateways = $stmt->fetchAll();
} catch(PDOException $e) {
    $gateways = [];
}

// Get bKash automatic gateway config
try {
    $stmt = $pdo->prepare("SELECT * FROM payment_gateways WHERE gateway_type = 'bkash' AND type = 'automatic'");
    $stmt->execute();
    $bkashConfig = $stmt->fetch();
} catch(PDOException $e) {
    $bkashConfig = null;
}

$flash = getFlash();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Gateways - Prime Institute Admin</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="admin-style.css">
</head>
<body>
    <?php include 'includes/header.php'; ?>
    
    <div class="admin-container">
        <?php include 'includes/sidebar.php'; ?>
        
        <main class="main-content">
            <div class="page-header">
                <h1>💳 Payment Gateways</h1>
                <p>Manage payment methods</p>
            </div>
            
            <?php if ($flash): ?>
            <div class="alert alert-<?php echo $flash['type']; ?>">
                <?php echo htmlspecialchars($flash['message']); ?>
            </div>
            <?php endif; ?>
            
            <!-- bKash Automatic API Configuration -->
            <div class="content-card" style="margin-bottom: 20px; border-left: 4px solid #e2136e;">
                <h3>🔷 bKash Automatic Payment Gateway (API)</h3>
                <p style="color: #9ca3af; margin-bottom: 20px;">Configure bKash Payment Gateway API for automatic payment processing</p>
                
                <form method="POST">
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                        <div class="form-group">
                            <label>API Username *</label>
                            <input type="text" name="bkash_username" placeholder="bKash API Username" 
                                   value="<?php echo htmlspecialchars($bkashConfig['api_username'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label>API Password *</label>
                            <input type="password" name="bkash_password" placeholder="bKash API Password" 
                                   value="<?php echo htmlspecialchars($bkashConfig['api_password'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label>App Key *</label>
                            <input type="text" name="bkash_app_key" placeholder="bKash App Key" 
                                   value="<?php echo htmlspecialchars($bkashConfig['api_app_key'] ?? ''); ?>">
                        </div>
                        
                        <div class="form-group">
                            <label>App Secret *</label>
                            <input type="password" name="bkash_app_secret" placeholder="bKash App Secret" 
                                   value="<?php echo htmlspecialchars($bkashConfig['api_app_secret'] ?? ''); ?>">
                        </div>
                    </div>
                    
                    <div style="display: flex; gap: 30px; margin-top: 15px; margin-bottom: 20px;">
                        <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                            <input type="checkbox" name="bkash_sandbox" value="1" 
                                   <?php echo ($bkashConfig['is_sandbox'] ?? 1) ? 'checked' : ''; ?>
                                   style="width: 18px; height: 18px;">
                            <span>Sandbox Mode (Testing)</span>
                        </label>
                        
                        <label style="display: flex; align-items: center; gap: 8px; cursor: pointer;">
                            <input type="checkbox" name="bkash_status" value="1" 
                                   <?php echo ($bkashConfig['status'] ?? 0) ? 'checked' : ''; ?>
                                   style="width: 18px; height: 18px;">
                            <span>Enable Gateway</span>
                        </label>
                    </div>
                    
                    <div style="background: #1e293b; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                        <p style="margin: 0; color: #94a3b8; font-size: 14px;">
                            <strong>📝 Note:</strong> To get bKash API credentials, you need to register as a bKash Merchant at 
                            <a href="https://developer.bka.sh/" target="_blank" style="color: #e2136e;">developer.bka.sh</a>. 
                            Enable Sandbox Mode for testing before going live.
                        </p>
                    </div>
                    
                    <button type="submit" name="save_bkash_config" class="btn btn-primary" style="background: #e2136e;">
                        💾 Save bKash Configuration
                    </button>
                </form>
            </div>
            
            <div class="content-card" style="margin-bottom: 20px;">
                <h3>➕ Add New Payment Gateway</h3>
                <form method="POST" style="margin-top: 20px;">
                    <div style="display: grid; grid-template-columns: 1fr 1fr; gap: 15px;">
                        <div class="form-group">
                            <label>Gateway Name *</label>
                            <input type="text" name="name" placeholder="e.g., Nagad, Rocket" required>
                        </div>
                        
                        <div class="form-group">
                            <label>Gateway Type</label>
                            <select name="gateway_type">
                                <option value="bkash">bKash</option>
                                <option value="nagad">Nagad</option>
                                <option value="rocket">Rocket</option>
                                <option value="bank">Bank Transfer</option>
                                <option value="other">Other</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Account Number *</label>
                            <input type="text" name="account_number" placeholder="01XXXXXXXXX" required>
                        </div>
                        
                        <div class="form-group">
                            <label>Account Name</label>
                            <input type="text" name="account_name" placeholder="Account holder name">
                        </div>
                        
                        <div class="form-group">
                            <label>Account Type</label>
                            <select name="account_type">
                                <option value="Personal">Personal</option>
                                <option value="Merchant">Merchant</option>
                                <option value="Agent">Agent</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label>Type</label>
                            <select name="type">
                                <option value="manual">Manual</option>
                                <option value="automatic" disabled>Automatic (Coming Soon)</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Payment Instructions</label>
                        <textarea name="instructions" rows="3" placeholder="Step-by-step payment instructions for customers"></textarea>
                    </div>
                    
                    <button type="submit" name="add_gateway" class="btn btn-primary">Add Gateway</button>
                </form>
            </div>
            
            <div class="content-card">
                <h3>📋 All Payment Gateways</h3>
                <div class="table-container">
                    <?php if (empty($gateways)): ?>
                        <p style="text-align: center; padding: 40px; color: #9ca3af;">No payment gateways configured</p>
                    <?php else: ?>
                    <table>
                        <thead>
                            <tr>
                                <th>Name</th>
                                <th>Type</th>
                                <th>Account Number</th>
                                <th>Account Name</th>
                                <th>Account Type</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($gateways as $gateway): ?>
                            <tr>
                                <td><strong><?php echo htmlspecialchars($gateway['name']); ?></strong></td>
                                <td>
                                    <span class="status-badge status-<?php echo $gateway['type']; ?>">
                                        <?php echo ucfirst($gateway['type']); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($gateway['account_number'] ?? '-'); ?></td>
                                <td><?php echo htmlspecialchars($gateway['account_name'] ?? '-'); ?></td>
                                <td><?php echo htmlspecialchars($gateway['account_type'] ?? '-'); ?></td>
                                <td>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="gateway_id" value="<?php echo $gateway['id']; ?>">
                                        <input type="hidden" name="status" value="<?php echo $gateway['status'] ? 0 : 1; ?>">
                                        <button type="submit" name="toggle_status" class="btn btn-sm <?php echo $gateway['status'] ? 'btn-success' : 'btn-secondary'; ?>">
                                            <?php echo $gateway['status'] ? 'Active' : 'Inactive'; ?>
                                        </button>
                                    </form>
                                </td>
                                <td>
                                    <?php if ($gateway['type'] === 'manual'): ?>
                                    <form method="POST" style="display: inline;">
                                        <input type="hidden" name="gateway_id" value="<?php echo $gateway['id']; ?>">
                                        <button type="submit" name="delete_gateway" class="btn btn-danger btn-sm" 
                                                onclick="return confirm('Delete this gateway?')">Delete</button>
                                    </form>
                                    <?php else: ?>
                                    <span style="color: #9ca3af;">System Gateway</span>
                                    <?php endif; ?>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>
</body>
</html>
